// SPDX-License-Identifier: LGPL-2.1-or-later
//
// SPDX-FileCopyrightText: 2011 Guillaume Martres <smarter@ubuntu.com>
//

#include "SatellitesConfigModel.h"

#include "SatellitesConfigLeafItem.h"

namespace Marble
{

SatellitesConfigModel::SatellitesConfigModel(QObject *parent)
    : QAbstractItemModel(parent)
    , m_rootItem(new SatellitesConfigNodeItem(QString()))
{
}

SatellitesConfigModel::~SatellitesConfigModel()
{
    delete m_rootItem;
}

QStringList SatellitesConfigModel::idList() const
{
    return m_rootItem->data(0, SatellitesConfigAbstractItem::IdListRole).toStringList();
}

QStringList SatellitesConfigModel::fullIdList() const
{
    return m_rootItem->data(0, SatellitesConfigAbstractItem::FullIdListRole).toStringList();
}

QStringList SatellitesConfigModel::urlList() const
{
    return m_rootItem->data(0, SatellitesConfigAbstractItem::UrlListRole).toStringList();
}

void SatellitesConfigModel::loadSettings(const QHash<QString, QVariant> &settings)
{
    m_rootItem->loadSettings(settings);
}

void SatellitesConfigModel::appendChild(SatellitesConfigAbstractItem *child)
{
    m_rootItem->appendChild(child);
}

void SatellitesConfigModel::clear()
{
    m_rootItem->clear();
}

QVariant SatellitesConfigModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid()) {
        return {};
    }

    auto item = static_cast<SatellitesConfigAbstractItem *>(index.internalPointer());
    return item->data(index.column(), role);
}

bool SatellitesConfigModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    auto item = static_cast<SatellitesConfigAbstractItem *>(index.internalPointer());

    bool success = item->setData(index.column(), role, value);

    if (success) {
        QModelIndex parentCellIndex = this->index(index.parent().row(), index.column(), index.parent().parent());
        Q_EMIT dataChanged(parentCellIndex, parentCellIndex);
    }

    return success;
}

int SatellitesConfigModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    // TODO: enable second column
    return 1;
}

int SatellitesConfigModel::rowCount(const QModelIndex &parent) const
{
    if (parent.column() > 0) {
        return 0;
    }

    SatellitesConfigAbstractItem *parentItem = nullptr;
    if (!parent.isValid()) {
        parentItem = m_rootItem;
    } else {
        parentItem = static_cast<SatellitesConfigAbstractItem *>(parent.internalPointer());
    }

    return parentItem->childrenCount();
}

QModelIndex SatellitesConfigModel::parent(const QModelIndex &child) const
{
    if (!child.isValid()) {
        return {};
    }

    auto childItem = static_cast<SatellitesConfigAbstractItem *>(child.internalPointer());
    SatellitesConfigAbstractItem *parentItem = childItem->parent();

    if (parentItem == m_rootItem) {
        return {};
    }

    return createIndex(parentItem->row(), 0, parentItem);
}

QModelIndex SatellitesConfigModel::index(int row, int column, const QModelIndex &parent) const
{
    if (!hasIndex(row, column, parent)) {
        return {};
    }

    SatellitesConfigAbstractItem *parentItem = nullptr;
    if (!parent.isValid()) {
        parentItem = m_rootItem;
    } else {
        parentItem = static_cast<SatellitesConfigAbstractItem *>(parent.internalPointer());
    }

    SatellitesConfigAbstractItem *childItem = parentItem->childAt(row);

    if (!childItem) {
        return {};
    }

    return createIndex(row, column, childItem);
}

QVariant SatellitesConfigModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if (orientation != Qt::Horizontal || role != Qt::DisplayRole) {
        return {};
    }

    switch (section) {
    case 0: {
        return QVariant(tr("Catalogs"));
    }
    default: {
        return {};
    }
    }
}

Qt::ItemFlags SatellitesConfigModel::flags(const QModelIndex &index) const
{
    if (!index.isValid()) {
        return {};
    }

    auto item = static_cast<SatellitesConfigAbstractItem *>(index.internalPointer());
    return item->flags();
}

SatellitesConfigNodeItem *SatellitesConfigModel::rootItem() const
{
    return m_rootItem;
}

} // namespace Marble

#include "moc_SatellitesConfigModel.cpp"
